import requests
import json
import typing
from functools import partial


from Deadline.Scripting import ClientUtils, MonitorUtils, RepositoryUtils
from Deadline.Jobs import Job, JobStatus

from DeadlineUI.Controls.Scripting.DeadlineScriptDialog import DeadlineScriptDialog
from ThinkboxUI.Controls.Scripting.ButtonControl import ButtonControl

def __main__():
    global scriptDialog

    scriptDialog = RestartDialog()
    # Add control buttons
    scriptDialog.AddGrid()
    scriptDialog.AddHorizontalSpacerToGrid( "HSpacer", 0, 0 )
    selectAllTasks = scriptDialog.AddSelectionControlToGrid( "all_tasks", "CheckBoxControl", True, "All Tasks", 0, 0, "All Tasks", expand=False )
    selectAllTasks.ValueModified.connect( SetOptionsForAllTasks )
    selectStoppedTasks = scriptDialog.AddSelectionControlToGrid( "non_completed_tasks", "CheckBoxControl", False, "Stopped Tasks", 1, 0, "Stopped Tasks", expand=False )
    selectStoppedTasks.ValueModified.connect( setOptionsForTasksByState )
    selectQueuedTasks = scriptDialog.AddSelectionControlToGrid( "non_started_tasks", "CheckBoxControl", False, "Non-Started Tasks", 2, 0, "Non-Started Tasks", expand=False )
    selectQueuedTasks.ValueModified.connect( setOptionsForTasksByState )
    selectFailedTasks = scriptDialog.AddSelectionControlToGrid( "failed_tasks", "CheckBoxControl", False, "Failed Tasks", 3, 0, "Failed Tasks", expand=False )
    selectFailedTasks.ValueModified.connect( setOptionsForTasksByState )
    selectCompletedTasks = scriptDialog.AddSelectionControlToGrid( "completed_tasks", "CheckBoxControl", False, "Completed Tasks", 4, 0, "Completed Tasks", expand=False )
    selectCompletedTasks.ValueModified.connect( setOptionsForTasksByState )
    selectTaskRange = scriptDialog.AddSelectionControlToGrid( "selected_tasks", "CheckBoxControl", False, "Task Range", 5, 0, "Task Range", expand=False )
    selectTaskRange.ValueModified.connect( setOptionsForTaskRange )
    scriptDialog.AddControlToGrid( "taskRangeText", "TextControl", "1-2", 6, 0 )
    scriptDialog.SetEnabled("taskRangeText", False)
    scriptDialog.AddControlToGrid( "RangeInfo", "LabelControl", "Input format: start-end:step. End and step are optional.\nMultiple frames ranges can be comma separated.\nEx.1 or 1-5 or 5-10:2 or 3,5-10,20-50:2", 7, 0 )
    scriptDialog.SetEnabled("RangeInfo", False)

    okButton = scriptDialog.AddControlToGrid( "OkButton", "ButtonControl", "Yes", 6, 2, expand=False )
    okButton.clicked.connect( OKButtonClicked )
    cancelButton = scriptDialog.AddControlToGrid( "CancelButton", "ButtonControl", "No", 6, 3, expand=False )
    cancelButton.clicked.connect( CancelButtonClicked )
    scriptDialog.EndGrid()

    SetOptionsForAllTasks()
    scriptDialog.ShowDialog( True )


def SetOptionsForAllTasks():
    selectAllTasksEnabled = scriptDialog.GetValue( "all_tasks" )
    if selectAllTasksEnabled:
        scriptDialog.SetEnabled("non_completed_tasks", False)
        scriptDialog.SetEnabled("non_started_tasks", False)
        scriptDialog.SetEnabled("failed_tasks", False)
        scriptDialog.SetEnabled("completed_tasks", False)
        scriptDialog.SetEnabled("selected_tasks", False)
        scriptDialog.SetEnabled("taskRangeText", False)
        scriptDialog.SetEnabled("RangeInfo", False)
    else:
        scriptDialog.SetEnabled("non_completed_tasks", True)
        scriptDialog.SetEnabled("non_started_tasks", True)
        scriptDialog.SetEnabled("failed_tasks", True)
        scriptDialog.SetEnabled("completed_tasks", True)
        scriptDialog.SetEnabled("selected_tasks", True)
        scriptDialog.SetEnabled("taskRangeText", False)
        scriptDialog.SetEnabled("RangeInfo", False)

def setOptionsForTasksByState():
    ncTasksEnabled = scriptDialog.GetValue( "non_completed_tasks" )
    nsTasksEnabled = scriptDialog.GetValue( "non_started_tasks" )
    failedTasksEnabled = scriptDialog.GetValue( "failed_tasks" )
    completedTasksEnabled = scriptDialog.GetValue( "completed_tasks" )

    if ncTasksEnabled or nsTasksEnabled or failedTasksEnabled or completedTasksEnabled:
        scriptDialog.SetEnabled("all_tasks", False)
        scriptDialog.SetEnabled("selected_tasks", False)
        scriptDialog.SetEnabled("taskRangeText", False)
        scriptDialog.SetEnabled("RangeInfo", False)
    else:
        scriptDialog.SetEnabled("all_tasks", True)
        scriptDialog.SetEnabled("selected_tasks", True)
        scriptDialog.SetEnabled("taskRangeText", False)
        scriptDialog.SetEnabled("RangeInfo", False)

def setOptionsForTaskRange():
    taskRangeEnabled = scriptDialog.GetValue( "selected_tasks" )

    if taskRangeEnabled:
        scriptDialog.SetEnabled("all_tasks", False)
        scriptDialog.SetEnabled("non_completed_tasks", False)
        scriptDialog.SetEnabled("non_started_tasks", False)
        scriptDialog.SetEnabled("failed_tasks", False)
        scriptDialog.SetEnabled("completed_tasks", False)
        scriptDialog.SetEnabled("taskRangeText", True)
        scriptDialog.SetEnabled("RangeInfo", True)
    else:
        scriptDialog.SetEnabled("all_tasks", True)
        scriptDialog.SetEnabled("non_completed_tasks", True)
        scriptDialog.SetEnabled("non_started_tasks", True)
        scriptDialog.SetEnabled("failed_tasks", True)
        scriptDialog.SetEnabled("completed_tasks", True)
        scriptDialog.SetEnabled("taskRangeText", False)
        scriptDialog.SetEnabled("RangeInfo", False)

def OKButtonClicked():
    # type: () -> None
    super( RestartDialog, scriptDialog ).accept()

    try:
        selectedJobs = MonitorUtils.GetSelectedJobs()
        job = selectedJobs[0]
        print("JobId: %s" % job.ExtraInfo1)
        print("ProjectId: %s" % job.ExtraInfo0)
        job_tasks = RepositoryUtils.GetJobTasks(job, True)

        restart_options = []
        restart_task_ids = []

        selectAllTasksEnabled = super( RestartDialog, scriptDialog ).GetValue( "all_tasks" )
        selectTaskRangeEnabled = super( RestartDialog, scriptDialog ).GetValue( "selected_tasks" )
        if selectAllTasksEnabled:
            restart_options.append("all_tasks")
            restart_task_ids = job.FramesList
            dl_restart_task_ids = gm_restart_task_ids = restart_task_ids
        elif selectTaskRangeEnabled:
            restart_options.append("selected_tasks")
            task_ranges = super( RestartDialog, scriptDialog ).GetValue( "taskRangeText" )

            gm_restart_task_list = [] 
            restart_task_list = task_ranges.split(",")           
            for tr in restart_task_list:
                tr = tr.replace("-"," ")
                tr = tr.replace(":"," ")
                gm_restart_task_list.append(tr)

            gm_restart_task_ids = ",".join(gm_restart_task_list)
            dl_restart_task_ids = task_ranges
        else:
            selectStoppedTasksEnabled = super( RestartDialog, scriptDialog ).GetValue( "non_completed_tasks" )
            if selectStoppedTasksEnabled:
                restart_options.append("non_completed_tasks")
                nc_task_ids = GetTaskIdsByState(job_tasks, ["Suspended"])
                restart_task_ids.extend(nc_task_ids)
            
            selectQueuedTasksEnabled = super( RestartDialog, scriptDialog ).GetValue( "non_started_tasks" )
            if selectQueuedTasksEnabled:
                restart_options.append("non_started_tasks")
                ns_task_ids = GetTaskIdsByState(job_tasks, ["Queued"])
                restart_task_ids.extend(ns_task_ids)
            
            selectFailedEnabled = super( RestartDialog, scriptDialog ).GetValue( "failed_tasks" )
            if selectFailedEnabled:
                restart_options.append("failed_tasks")
                failed_task_ids = GetTaskIdsByState(job_tasks, ["Failed"])
                restart_task_ids.extend(failed_task_ids)

            selectCompletedEnabled = super( RestartDialog, scriptDialog ).GetValue( "completed_tasks" )
            if selectCompletedEnabled:
                restart_options.append("completed_tasks")
                completed_task_ids = GetTaskIdsByState(job_tasks, ["Completed"])
                restart_task_ids.extend(completed_task_ids)

            restart_task_ids = list(dict.fromkeys(restart_task_ids))
            restart_task_ids.sort()

            gm_restart_task_ids = dl_restart_task_ids = ",".join(map(str, restart_task_ids))

        print("%%%%%%%%%%%%%%%%%%%%%%%%%")
        print(gm_restart_task_ids, dl_restart_task_ids)
        RestartGridMarketsJob(job, restart_options, gm_restart_task_ids, dl_restart_task_ids)
    except Exception as e:
        super( RestartDialog, scriptDialog ).ShowMessageBox( "Error restarting job: %s" % str(e), "Error" )

def CancelButtonClicked():
    # type: () -> None
    super( RestartDialog, scriptDialog ).reject()

def GetTaskIdsByState(task_list, status_list):
    matching_task_ids = []
    for task in task_list:
        if (str(task.Status) in status_list):
            matching_task_ids.append(int(task.TaskFrameString.split("-")[0]))
    return matching_task_ids

def RestartGridMarketsJob(job, restart_options, gm_restart_task_ids, dl_restart_task_ids):
    gmJobId = job.ExtraInfo1
    gmProjectId = job.ExtraInfo0

    print("Restarting GridMarkets job...")    
    url = "http://localhost:8090/submissions/%s/jobs/%s?action=restart" % (gmProjectId, gmJobId)
    headers = {"Content-Type": "application/json"}
    data = {
                "params" : {
                    "restart_options": restart_options,
                },
                "deadline": {
                    "ProjectName": job.Name.split(" - ")[0],
                    "JobName": job.Name.split(" - ")[1],
                    "Frames": dl_restart_task_ids
                }
            }
    
    if ("selected_tasks" in restart_options):
        data["params"]["frames"] = gm_restart_task_ids

    try:
        response = requests.post(url, headers=headers, data=json.dumps(data))
        if response.status_code == 200 or response.status_code == 201  or response.status_code == 202:
            print(f"Status Code: {response.status_code}")
            print(f"Response: {response.json()}")
            super( RestartDialog, scriptDialog ).ShowMessageBox( "Restarting job", "Info" )
        else:
            if response.status_code == 401:
                raise Exception("Please login to Envoy.")
            else:
                raise Exception(f"{response.json()}")
    except Exception as e:
        print(f"Error restarting job: {str(e)}")
        super( RestartDialog, scriptDialog ).ShowMessageBox( "Error restarting job: %s" % str(e), "Error" )
        
########################################################################
## Subclass of DeadlineScriptDialog for the UI
########################################################################
class RestartDialog( DeadlineScriptDialog ):

    def __init__( self, parentAppName="", parent=None ):
        # type: (str, Optional[Any]) -> None
        super( RestartDialog, self ).__init__( parent )
        self.SetTitle( "Restart Job" )
        self.AddControl( "StopText", "LabelControl", "Are you sure, do you want to restart this job?", 480, -1 )

    @classmethod
    def ShowMessageBox( cls, message, title="Info" ):
        # type: (str, str) -> None
        if cls is not None:
            ClientUtils.ShowMessageBox( message, title )
        else:
            print( "{}: {}".format( title, message ) )

########################################################################
## Globals
########################################################################
# down here to ensure all functions/classes exist
if typing.TYPE_CHECKING:
    scriptDialog = RestartDialog()
else:
    scriptDialog = None

